<?php

declare(strict_types=1);

/**
 * This file is part of CodeIgniter Shield.
 *
 * (c) CodeIgniter Foundation <admin@codeigniter.com>
 *
 * For the full copyright and license information, please view
 * the LICENSE file that was distributed with this source code.
 */

namespace Tests\Collectors;

use CodeIgniter\Shield\Auth as ShieldAuth;
use CodeIgniter\Shield\Authentication\Authenticators\Session;
use CodeIgniter\Shield\Collectors\Auth;
use CodeIgniter\Shield\Entities\User;
use CodeIgniter\Shield\Models\UserModel;
use CodeIgniter\Test\DatabaseTestTrait;
use Tests\Support\TestCase;

/**
 * @internal
 */
final class AuthTest extends TestCase
{
    use DatabaseTestTrait;

    protected $namespace;
    protected $refresh = true;
    private User $user;
    private Auth $collector;

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = fake(UserModel::class, ['username' => 'John Smith']);

        $this->collector = new Auth();
    }

    public function testDisplayNotLoggedIn(): void
    {
        $output = $this->collector->display();

        $this->assertStringContainsString('Not logged in', $output);
    }

    public function testDisplayLoggedIn(): void
    {
        $authenticator = service('auth')->getAuthenticator();
        assert($authenticator instanceof Session);
        $authenticator->login($this->user);
        $this->user->addGroup('admin', 'beta');
        $this->user->addPermission('users.create', 'users.edit');

        $output = $this->collector->display();

        $this->assertStringContainsString('Current Use', $output);
        $this->assertStringContainsString('<td>Username</td><td>John Smith</td>', $output);
        $this->assertStringContainsString('<td>Groups</td><td>admin, beta</td>', $output);
        $this->assertStringContainsString('<td>Permissions</td><td>users.create, users.edit</td>', $output);
    }

    public function testDisplayNotLoggedInAfterLogout(): void
    {
        $authenticator = service('auth')->getAuthenticator();
        assert($authenticator instanceof Session);
        $authenticator->login($this->user);

        $authenticator->logout();

        $output = $this->collector->display();
        $this->assertStringContainsString('Not logged in', $output);
    }

    public function testGetTitleDetails(): void
    {
        $output = $this->collector->getTitleDetails();

        $this->assertStringContainsString(ShieldAuth::SHIELD_VERSION, $output);
        $this->assertStringContainsString(Session::class, $output);
    }

    public function testGetBadgeValueReturnsUserId(): void
    {
        /** @var Session $authenticator */
        $authenticator = service('auth')->getAuthenticator();
        $authenticator->login($this->user);

        $output = (string) $this->collector->getBadgeValue();

        $this->assertStringContainsString('1', $output);
    }
}
