<?php

namespace App\Controllers;

use CodeIgniter\Controller;
use App\Models\UndepositedFundsModel;
use App\Libraries\undeposited_funds\UndepositedFundsLibrary;

class UndepositedFunds extends Controller
{
    public function index()
    {
        if (ENVIRONMENT === 'development') {
            cache()->clean();
        }

        if (! service('auth')->loggedIn()) {
            return redirect()->to('/login');
        }
        $user = service('auth')->user();
        if ((int) $user->is_admin !== 1) {
            return redirect()->to('/')->with('error', 'Access denied: Admins only.');
        }

        $session     = session();
        $realmId     = $session->get('realmid');
        $accessToken = $session->get('access_token');
        $isConnected = ($realmId && $accessToken);

        $req      = service('request');
        $fromDate = $req->getGetPost('from_date');
        $toDate   = $req->getGetPost('to_date');
        $action   = $req->getPost('action');

        $firstOfMonth = date('Y-m-01');
        $today        = date('Y-m-d');
        if (empty($fromDate)) {
            $fromDate = $firstOfMonth;
        }
        if (empty($toDate)) {
            $toDate = $today;
        }

        // Use the library for pulling data
        $ufLib = new UndepositedFundsLibrary();

        // Pull data if requested
        if ($action === 'pull_undeposited' && $isConnected && $fromDate && $toDate) {
            try {
                $ufModel = new UndepositedFundsModel();
                $ufLib->pullPayments($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $ufLib->pullSalesReceipts($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $ufLib->pullRefundReceipts($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $ufLib->pullCreditMemos($realmId, $accessToken, $fromDate, $toDate, $ufModel);
                $ufLib->pullJournalEntries($realmId, $accessToken, $fromDate, $toDate, $ufModel);

                $ufLib->pullDeposits($realmId, $accessToken, $fromDate, $toDate);
                $this->crossReferenceDeposits();
            } catch (\Throwable $e) {
                $error = 'Error while pulling data: ' . $e->getMessage();
            }
        }
        if ($action === 'pull_deposits' && $isConnected && $fromDate && $toDate) {
            try {
                $ufLib->pullDeposits($realmId, $accessToken, $fromDate, $toDate);
                $this->crossReferenceDeposits();
            } catch (\Throwable $e) {
                $error = 'Error while pulling deposits: ' . $e->getMessage();
            }
        }

        $ufModel  = new UndepositedFundsModel();
        $limitVal = $req->getGetPost('limit') ?: '100';
        $all      = ($limitVal === 'ALL');
        $results  = [];
        $pager    = null;

        if ($all) {
            $results = $ufModel->where('txn_date >=', $fromDate)
                               ->where('txn_date <=', $toDate)
                               ->orderBy('txn_date', 'ASC')
                               ->findAll();
        } else {
            $perPage = (int) $limitVal;
            if ($perPage < 1) {
                $perPage = 100;
            }
            $ufModel->where('txn_date >=', $fromDate)
                    ->where('txn_date <=', $toDate)
                    ->orderBy('txn_date', 'ASC');
            $results = $ufModel->paginate($perPage, 'ufGroup');
            $pager   = $ufModel->pager;
        }

        $data = [
            'isConnected' => $isConnected,
            'error'       => $error ?? null,
            'from_date'   => $fromDate,
            'to_date'     => $toDate,
            'results'     => $results,
            'pager'       => $pager,
            'limitVal'    => $limitVal
        ];

        return view('undepositedfunds/index', $data);
    }

    // Universal AJAX loader for both left and middle panes
    public function loadView()
    {
        $req     = service('request');
        $pane    = $req->getGet('pane'); // 'left' or 'middle'
        $view    = $req->getGet('view'); // e.g. 'make_bank_deposit', 'main_view_selector', etc.
        $fresh   = $req->getGet('fresh') ?? '0'; // '1' means clear session
        $session = session();

        // LEFT PANE: handle left pane requests directly
        if ($pane === 'left') {
            // When switching to the make bank deposit view
            if ($view === 'make_bank_deposit') {
                return view('undepositedfunds/left/make_bank_deposit');
            }
            // Default left pane (main selector)
            return view('undepositedfunds/left/main_view_selector');
        }

        // MIDDLE PANE: handle logic as before
        $realmId     = $session->get('realmid');
        $accessToken = $session->get('access_token');
        $isConnected = ($realmId && $accessToken);

        $limitVal = $req->getGet('limit') ?: '100';
        $all      = ($limitVal === 'ALL');
        $pager    = null;

        $fromDate = $req->getGet('from_date') ?? date('Y-m-01');
        $toDate   = $req->getGet('to_date')   ?? date('Y-m-d');

        // If user is NOT going to "make_bank_deposit", remove existing session array
        if ($view !== 'make_bank_deposit') {
            $session->remove('transaction_ids_selected');
        }

        $data = [
            'isConnected' => $isConnected,
            'error'       => '',
            'from_date'   => $fromDate,
            'to_date'     => $toDate,
            'limitVal'    => $limitVal,
            'pager'       => null,
            'results'     => [],
        ];

        // MIDDLE: partials for middle pane
        if ($view === 'pull') {
            return view('undepositedfunds/middle/pull_undeposited_funds', $data);
        }

        if ($view === 'make_bank_deposit') {
            // If fresh=1 => remove old selections
            if ($fresh === '1') {
                $session->remove('transaction_ids_selected');
            }

            $ufModel = new UndepositedFundsModel();
            if ($all) {
                $results = $ufModel->where('is_deposited', 0)
                                   ->orderBy('txn_date', 'ASC')
                                   ->findAll();
            } else {
                $perPage = (int) $limitVal;
                if ($perPage < 1) {
                    $perPage = 100;
                }
                $ufModel->where('is_deposited', 0)->orderBy('txn_date', 'ASC');
                $results = $ufModel->paginate($perPage, 'mbdGroup');
                $pager   = $ufModel->pager;
            }
            $data['results'] = $results;
            $data['pager']   = $pager;

            return view('undepositedfunds/middle/make_bank_deposit', $data);
        }

        // Otherwise default "manage"
        $ufModel = new UndepositedFundsModel();
        if ($all) {
            $results = $ufModel->where('txn_date >=', $fromDate)
                               ->where('txn_date <=', $toDate)
                               ->orderBy('txn_date', 'ASC')
                               ->findAll();
        } else {
            $perPage = (int) $limitVal;
            if ($perPage < 1) {
                $perPage = 100;
            }
            $ufModel->where('txn_date >=', $fromDate)
                    ->where('txn_date <=', $toDate)
                    ->orderBy('txn_date', 'ASC');
            $results = $ufModel->paginate($perPage, 'ufGroup');
            $pager   = $ufModel->pager;
        }

        $data['results'] = $results;
        $data['pager']   = $pager;

        return view('undepositedfunds/middle/manage_undeposited_funds', $data);
    }

    // ---------------------------------------------------
    // crossReferenceDeposits() kept here per your request
    // ---------------------------------------------------
    private function crossReferenceDeposits(): void
    {
        $db = \Config\Database::connect();
        $sql = "
            UPDATE undeposited_funds uf
            JOIN deposits_made_lines dml ON dml.linked_txn_id = uf.qb_txn_id
            JOIN deposits_made       dm  ON dm.id            = dml.deposits_made_id
            SET uf.is_deposited    = 1,
                uf.deposit_txn_id  = dm.qb_deposit_id,
                uf.deposit_date    = dm.deposit_date
            WHERE uf.is_deposited  = 0
        ";
        $db->query($sql);
    }

    // ---------------------------------------------------
    // updateSelected() now only updates the session data
    // ---------------------------------------------------
    public function updateSelected()
    {
        if (! $this->request->isAJAX()) {
            return $this->response->setJSON(['error' => 'AJAX only'])->setStatusCode(400);
        }
        $body = $this->request->getJSON(true);
        $ufId    = $body['ufId']    ?? null;
        $checked = $body['checked'] ?? false;

        if (! $ufId) {
            return $this->response->setJSON(['error' => 'No ufId'])->setStatusCode(400);
        }

        $session = session();
        $selected = $session->get('transaction_ids_selected') ?? [];

        if ($checked) {
            if (! in_array($ufId, $selected)) {
                $selected[] = $ufId;
            }
        } else {
            $selected = array_filter($selected, function ($id) use ($ufId) {
                return $id != $ufId;
            });
        }
        $session->set('transaction_ids_selected', $selected);

        // No sum logic now, just return success
        return $this->response->setJSON(['success' => true]);
    }

    // DEPRECATED: These can be replaced by loadView's left pane logic if you want.
    public function leftMakeBankDeposit()
    {
        return view('undepositedfunds/left/make_bank_deposit');
    }

    public function clearSelected()
    {
        session()->remove('transaction_ids_selected');
        return view('undepositedfunds/left/main_view_selector');
    }
}
