<?php

namespace App\Models;

use CodeIgniter\Model;

class UndepositedFundsModel extends Model
{
    protected $table      = 'undeposited_funds';
    protected $primaryKey = 'id';

    protected $allowedFields = [
        'qb_txn_id',
        'txn_type',
        'txn_date',
        'customer_name',
        'payment_method',
        'amount',
        'deposit_to_account',
        'memo',
        'reference_number',
        'is_deposited',
        'deposit_txn_id',
        'deposit_date',
        'journal_reference',
        'created_at',
        // JournalEntry extras
        'debit_amount',
        'credit_amount',
        'je_description',
        'je_account_name',
        'je_line_name',
        // Additional columns
        'payment_ref_num',
        'refund_from_account',
        'unapplied_amt',
        'exchange_rate',
        'currency_ref_name',
        'ar_account_ref_name',
        'line_items_json',
    ];

    /**
     * Insert or update a transaction record in undeposited_funds
     * for Payment | SalesReceipt | RefundReceipt | CreditMemo | JournalEntry.
     */
    public function upsertUndepositedRecord(array $txn, string $txnType): void
    {
        $qbId = $txn['Id'] ?? null;
        if (! $qbId) {
            return;
        }

        // Common fields for all transaction types
        $data = [
            'qb_txn_id'        => $qbId,
            'txn_type'         => $txnType,
            'txn_date'         => $txn['TxnDate']            ?? null,
            'memo'             => $txn['PrivateNote']        ?? '',
            'reference_number' => $txn['DocNumber']
                                   ?? ($txn['TxnReferenceNumber'] ?? ''),
            'customer_name'    => $txn['CustomerRef']['name'] ?? '',
            'payment_method'   => $txn['PaymentMethodRef']['name'] ?? '',
            'amount'           => $txn['TotalAmt']           ?? 0,
            'is_deposited'     => 0,
        ];

        // Multi-currency
        if (! empty($txn['ExchangeRate'])) {
            $data['exchange_rate'] = $txn['ExchangeRate'];
        }
        if (! empty($txn['CurrencyRef']['name'])) {
            $data['currency_ref_name'] = $txn['CurrencyRef']['name'];
        }

        switch ($txnType) {
            /* ───────────────────────────────
               1) Payment (Receive Payment)
               ─────────────────────────────── */
            case 'Payment':
                $data['payment_ref_num'] = $txn['PaymentRefNum'] ?? '';
                $data['unapplied_amt']   = $txn['UnappliedAmt']  ?? 0;

                if (! empty($txn['ARAccountRef']['name'])) {
                    $data['ar_account_ref_name'] = $txn['ARAccountRef']['name'];
                }
                if (! empty($txn['Line'])) {
                    $data['line_items_json'] = json_encode($txn['Line']);
                }

                // “Undeposited Funds” may be in DepositToAccountRef or BankAccountRef.
                $depName  = $txn['DepositToAccountRef']['name'] ?? '';
                $bankName = $txn['BankAccountRef']['name']      ?? '';

                if ($depName === '' && $bankName === '') {
                    // Some payments omit both refs — treat them as Undeposited Funds.
                    $data['deposit_to_account'] = 'Undeposited Funds';
                } elseif (stripos($depName,  'Undeposited') !== false) {
                    $data['deposit_to_account'] = $depName;
                } elseif (stripos($bankName, 'Undeposited') !== false) {
                    $data['deposit_to_account'] = $bankName;
                } else {
                    // Not an undeposited payment
                    return;
                }

                // NEW RULE: Only proceed if deposit_to_account is exactly "Undeposited Funds".
                if ($data['deposit_to_account'] !== 'Undeposited Funds') {
                    return;
                }
                break;

            /* ───────────────────────────────
               2) SalesReceipt
               ─────────────────────────────── */
            case 'SalesReceipt':
                if (! empty($txn['Line'])) {
                    $data['line_items_json'] = json_encode($txn['Line']);
                }
                $depName = $txn['DepositToAccountRef']['name'] ?? '';
                if (stripos($depName, 'Undeposited') === false) {
                    return;
                }
                $data['deposit_to_account'] = $depName;
                break;

            /* ───────────────────────────────
               3) RefundReceipt
               ─────────────────────────────── */
            case 'RefundReceipt':
                $refundAcct = $txn['RefundFromAccountRef']['name'] ?? '';
                $depName    = $txn['DepositToAccountRef']['name']  ?? '';

                if (stripos($refundAcct, 'Undeposited') !== false) {
                    $data['refund_from_account'] = $refundAcct;
                    $data['deposit_to_account']  = $refundAcct;
                } elseif (stripos($depName, 'Undeposited') !== false) {
                    $data['refund_from_account'] = $depName;
                    $data['deposit_to_account']  = $depName;
                } else {
                    return;
                }

                if (! empty($txn['Line'])) {
                    $data['line_items_json'] = json_encode($txn['Line']);
                }
                break;

            /* ───────────────────────────────
               4) CreditMemo
               ─────────────────────────────── */
            case 'CreditMemo':
                $depName = $txn['DepositToAccountRef']['name'] ?? '';
                if (stripos($depName, 'Undeposited') === false) {
                    return;
                }
                $data['deposit_to_account'] = $depName;

                if (! empty($txn['Line'])) {
                    $data['line_items_json'] = json_encode($txn['Line']);
                }
                break;

            /* ───────────────────────────────
               5) JournalEntry
               ─────────────────────────────── */
            case 'JournalEntry':
                $data['journal_reference'] = $txn['DocNumber'] ?? '';
                $debit  = 0.0;
                $credit = 0.0;
                $desc   = '';
                $acct   = '';
                $name   = '';

                foreach ($txn['Line'] ?? [] as $ln) {
                    $det = $ln['JournalEntryLineDetail'] ?? null;
                    if (! $det) {
                        continue;
                    }
                    $posting  = $det['PostingType']        ?? '';
                    $amt      = (float) ($ln['Amount']     ?? 0);
                    $acctNm   = $det['AccountRef']['name'] ?? '';
                    $lineDesc = $ln['Description']         ?? '';
                    $entity   = $det['Entity']['EntityRef']['name']
                                ?? ($det['Entity']['Name'] ?? '');

                    if (stripos($acctNm, 'Undeposited Funds') !== false) {
                        if ($posting === 'Debit')  { $debit  += $amt; }
                        if ($posting === 'Credit') { $credit += $amt; }

                        $desc = $desc ?: $lineDesc;
                        $acct = $acct ?: $acctNm;
                        $name = $name ?: $entity;
                    }
                }

                if ($debit == 0 && $credit == 0) {
                    return; // nothing hit Undeposited Funds
                }

                $data['debit_amount']    = $debit;
                $data['credit_amount']   = $credit;
                $data['je_description']  = $desc;
                $data['je_account_name'] = $acct;
                $data['je_line_name']    = $name;
                break;

            default:
                return;
        }

        // Upsert
        $existing = $this->where('qb_txn_id', $qbId)->first();
        if ($existing) {
            $this->update($existing['id'], $data);
        } else {
            $this->insert($data);
        }
    }

    /** Pull by date range for the grid */
    public function fetchByDateRange(string $from, string $to): array
    {
        return $this->where('txn_date >=', $from)
                    ->where('txn_date <=', $to)
                    ->orderBy('txn_date', 'ASC')
                    ->findAll();
    }
}
