<?php

namespace App\Libraries\undeposited_funds;

use App\Models\UndepositedFundsModel;

class UndepositedFundsLibrary
{
    /**
     * Check if the account name contains "undeposited" (case-insensitive).
     */
    public function strContainsUndeposited(?string $s): bool
    {
        return $s && stripos($s, 'undeposited') !== false;
    }

    /**
     * Run a raw query against the QuickBooks Online API.
     */
    public function runQuery(string $realm, string $tok, string $q): array
    {
        $url = "https://quickbooks.api.intuit.com/v3/company/{$realm}/query?query=" . urlencode($q);

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_HTTPHEADER     => [
                "Authorization: Bearer {$tok}",
                "Accept: application/json",
                "Content-Type: application/text",
            ],
            CURLOPT_RETURNTRANSFER => true,
        ]);
        $resp = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($code !== 200) {
            throw new \RuntimeException("QBO API error {$code}: {$resp}");
        }
        return json_decode($resp, true);
    }

    /**
     * Pull deposits from QBO and upsert into the model.
     */
    public function pullDeposits(string $realm, string $tok, string $from, string $to)
    {
        $ufModel = new UndepositedFundsModel();
        $start   = 1;
        $max     = 1000;
        do {
            $q = "
              SELECT *
              FROM Deposit
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['Deposit'] ?? [];
            foreach ($rows as $d) {
                $ufModel->upsertDeposit($d);
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    public function pullPayments(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM Payment
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['Payment'] ?? [];
            foreach ($rows as $p) {
                $ufModel->upsertUndepositedRecord($p, 'Payment');
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    public function pullSalesReceipts(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM SalesReceipt
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['SalesReceipt'] ?? [];
            foreach ($rows as $sr) {
                $acctName = $sr['DepositToAccountRef']['name'] ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($sr, 'SalesReceipt');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    public function pullRefundReceipts(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM RefundReceipt
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['RefundReceipt'] ?? [];
            foreach ($rows as $rr) {
                $acctName = $rr['RefundFromAccountRef']['name']
                            ?? $rr['DepositToAccountRef']['name']
                            ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($rr, 'RefundReceipt');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    public function pullCreditMemos(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM CreditMemo
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['CreditMemo'] ?? [];
            foreach ($rows as $cm) {
                $acctName = $cm['DepositToAccountRef']['name'] ?? '';
                if ($this->strContainsUndeposited($acctName)) {
                    $ufModel->upsertUndepositedRecord($cm, 'CreditMemo');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }

    public function pullJournalEntries(string $realm, string $tok, string $from, string $to, UndepositedFundsModel $ufModel)
    {
        $start = 1;
        $max   = 1000;
        do {
            $q = "
              SELECT *
              FROM JournalEntry
              WHERE TxnDate >= '{$from}' AND TxnDate <= '{$to}'
              ORDERBY Id
              STARTPOSITION {$start}
              MAXRESULTS {$max}";
            $rows = $this->runQuery($realm, $tok, $q)['QueryResponse']['JournalEntry'] ?? [];
            foreach ($rows as $je) {
                $hasUF = false;
                foreach ($je['Line'] ?? [] as $ln) {
                    $ref = $ln['JournalEntryLineDetail']['AccountRef'] ?? [];
                    if ($this->strContainsUndeposited($ref['name'] ?? '')) {
                        $hasUF = true;
                        break;
                    }
                }
                if ($hasUF) {
                    $ufModel->upsertUndepositedRecord($je, 'JournalEntry');
                }
            }
            $start += $max;
        } while (count($rows) === $max);
    }
}
